/*
* BufferSelector.java
* Copyright (c) 1999 Chris Laird
* Copyright (c) 2000, 2001, 2002 Andre Kaplan
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/


package bufferselector;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Container;
import java.awt.Font;
import java.awt.GridLayout;
import java.awt.Insets;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.awt.event.ComponentListener;
import java.awt.event.FocusAdapter;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.awt.event.MouseEvent;
import java.util.Enumeration;
import java.util.Hashtable;

import javax.swing.AbstractButton;
import javax.swing.BorderFactory;
import javax.swing.ButtonGroup;
import javax.swing.Icon;
import javax.swing.JPanel;
import javax.swing.JToggleButton;
import javax.swing.UIManager;
import javax.swing.JToolTip;

import org.gjt.sp.jedit.Buffer;
import org.gjt.sp.jedit.EBComponent;
import org.gjt.sp.jedit.EBMessage;
import org.gjt.sp.jedit.EditBus;
import org.gjt.sp.jedit.EditPane;
import org.gjt.sp.jedit.GUIUtilities;
import org.gjt.sp.jedit.View;
import org.gjt.sp.jedit.jEdit;
import org.gjt.sp.jedit.msg.BufferUpdate;
import org.gjt.sp.jedit.msg.EditPaneUpdate;
import org.gjt.sp.util.Log;


/**
* A <code>BufferSelector</code> is a container which holds toggle buttons which
* allow you to select a buffer. It has the same function as the Buffers menu.
* Each instance of <code>BufferSelector</code> is attached to a View.
*
* @author Andre Kaplan
* @author Chris Laird
* @version $Id: BufferSelector.java,v 1.18 2002/05/14 22:40:29 akaplan Exp $
*/
public class BufferSelector
extends JPanel
implements EBComponent
{
  public static final int TOP    = 0;
  public static final int BOTTOM = 1;
  
  /**
  * Records which <code>BufferSelector</code> is attached
  * to which <code>View</code>.
  */
  private static Hashtable bufferSelectors = new Hashtable();
  
  /** The margin that the buffer selector buttons will have */
  private static final Insets MARGIN = new Insets(0, 0, 0, 0);
  
  /* BufferSelector global properties */
  private static boolean defaultEnabled;
  private static int location;
  private static int minWidth;
  private static int maxWidth;
  private static int hgap;
  private static int vgap;
  
  /** The View that this BufferSelector is attached to */
  private View view;
  
  private Docker topDocker;
  private Docker bottomDocker;
  
  // Handlers
  private ComponentListener viewResizeHandler;
  private FocusListener textAreaFocusHandler;
  /** The ActionListener that listens to selector button presses */
  private ActionListener actionHandler;
  
  // Initialize the BufferSelector properties
  static {
    BufferSelector.getProperties();
  }
  
  
  /**
  * Create a new <code>BufferSelector</code> attached to the specified view.
  * @param view the <code>View</code> that this BufferSelector will be
  *        attached to
  */
  private BufferSelector(View view) {
    super();
    
    this.setBorder(
      BorderFactory.createCompoundBorder(
        BorderFactory.createCompoundBorder(
          BorderFactory.createEmptyBorder(4, 0, 0, 0),
          BorderFactory.createLoweredBevelBorder()
          ),
        BorderFactory.createEmptyBorder(2, 2, 1, 2)
        )
      );
    
    this.topDocker    = new ViewDocker(
      View.TOP_GROUP, View.BELOW_SEARCH_BAR_LAYER
      );
    this.bottomDocker = new ViewDocker(
      View.BOTTOM_GROUP, View.BELOW_STATUS_BAR_LAYER
      );
    
    this.view = view;
    this.viewResizeHandler    = new ViewResizeHandler();
    this.textAreaFocusHandler = new TextAreaFocusHandler();
    this.actionHandler        = new ActionHandler();
  }
  
  
  // EBComponent implementation
  public void handleMessage(EBMessage message) {
    if (message instanceof BufferUpdate) {
      BufferUpdate bu = (BufferUpdate)message;
      if (    (bu.getWhat() == BufferUpdate.CREATED)
        ||  (bu.getWhat() == BufferUpdate.CLOSED)
      ||  (bu.getWhat() == BufferUpdate.DIRTY_CHANGED)
      ) {
      this.update();
      this.view.invalidate();
      this.view.validate();
      this.view.repaint();
      }
    } else if (message instanceof EditPaneUpdate) {
      EditPaneUpdate epu = (EditPaneUpdate) message;
      View v = ((EditPane) epu.getSource()).getView();
      Log.log(Log.DEBUG, this, "***** EditPaneUpdate " + v);
      if (v == this.view) {
        if (epu.getWhat() == EditPaneUpdate.CREATED) {
          epu.getEditPane().getTextArea().addFocusListener(
            this.textAreaFocusHandler
            );
          this.view.repaint();
        } else if (epu.getWhat() == EditPaneUpdate.DESTROYED) {
          epu.getEditPane().getTextArea().removeFocusListener(
            this.textAreaFocusHandler
            );
          this.view.repaint();
        } else if (epu.getWhat() == EditPaneUpdate.BUFFER_CHANGED) {
          this.bufferChanged();
          // Funa edit
          this.update();
          this.view.invalidate();
          this.view.validate();
          this.view.repaint();
          
          // Funa edit
        } else if (epu.getWhat() == EditPaneUpdate.BUFFERSET_CHANGED) {
          this.update();
          this.view.invalidate();
          this.view.validate();
          this.view.repaint();
        }
      }
    }
  }
  
  
  /**
  * Update the <code>BufferSelector</code>.
  * This method is called by <code>View</code> objects when a buffer is
  * opened, closed or a new buffer is selected (either by one of the
  * selector buttons being pressed, or by one of the item in the Buffers
  * menu being chosen).
  */
  public void update() {
    // Get the list of buffers, and add a button for each one
    
    // Funa edit
    // Buffer[] buffers = jEdit.getBuffers();
    Buffer[] buffers = this.view.getEditPane().getBufferSet().getAllBuffers();
    int buffersCount = buffers.length;
    
    int minComponentsByLine = this.view.getSize().width / maxWidth;
    if (minComponentsByLine == 0) { minComponentsByLine = 1; }
    int maxComponentsByLine = this.view.getSize().width / minWidth;
    if (maxComponentsByLine == 0) { maxComponentsByLine = 1; }
    
    if (    (this.getLayout() == null)
      ||  !(this.getLayout() instanceof GridLayout)
    ) {
    this.setLayout(new GridLayout(0, minComponentsByLine, hgap, vgap));
    }
    
    GridLayout gl = (GridLayout) this.getLayout();
    
    gl.setRows(0);
    if (buffersCount < minComponentsByLine) {
      gl.setColumns(minComponentsByLine);
    } else if (
      (buffersCount >= minComponentsByLine)
      && (buffersCount <= maxComponentsByLine)
      ) {
    gl.setColumns(buffersCount);
      } else if (buffersCount > maxComponentsByLine) {
        // Get the minimun number of necessary rows to lay all buttons
        int rows = (
          (buffersCount / maxComponentsByLine)
          + ((buffersCount % maxComponentsByLine == 0) ? 0 : 1)
          );
        // Adjust the number of columns
        int cols = (
          (buffersCount / rows)
          + ((buffersCount % rows == 0) ? 0 : 1)
          );
        if (cols < minComponentsByLine) { cols = minComponentsByLine; }
        if (cols > maxComponentsByLine) { cols = maxComponentsByLine; }
        
        gl.setColumns(cols);
      }
      
      gl.setHgap(hgap);
      gl.setVgap(vgap);
      
      int fontSize = Integer.parseInt(
        jEdit.getProperty("buffer-selector.font-size")
        );
      
      // Remove all buttons from the selector, removing them as
      // ActionListeners and BufferListeners first
      this.removeAllButtons();
      ButtonGroup group = new ButtonGroup();
      // Stores the font to be used for the buttons
      Font buttonFont = null;
      
      for (int i = 0; i < buffersCount; i++) {
        Buffer buffer = buffers[i];
        
        // Use the buffer title as the text on the button
        String name = buffer.getName();
        JToggleButton toggle = new JToggleButton(name) {
          
          // Hack, needed because the tooltip in swing won't go away when
          // you move the mouse cursor over it. Worse than that - under
          // the windows look and feel it doesn't disappear (under jdk1.3)
          // or eats the mouse event (under jdk1.4) when it's pressed.
          public Point getToolTipLocation(MouseEvent event) {
            if (location == BufferSelector.TOP) {
              return new Point(0, this.getHeight() + 1);
            } else {
              JToolTip tip = this.createToolTip();
              tip.setTipText(this.getToolTipText());
              int prefTooltipHeight = tip.getPreferredSize().height;
              return new Point(0, -prefTooltipHeight - 1);
            }
          }
          
        };
        
        toggle.setMargin(MARGIN);
        toggle.setHorizontalAlignment(JToggleButton.LEFT);
        
        // When the first button is created, read its font
        if (buttonFont == null) {
          Font f = toggle.getFont();
          buttonFont = new Font(f.getName(), Font.PLAIN, fontSize);
        }
        toggle.setFont(buttonFont);
        
        if (buffer.isReadOnly()) {
          toggle.setForeground(Color.gray);
        }
        
        // Set up the tooltip (the full path to the buffer)
        String tooltip = buffer.getPath();
        toggle.setToolTipText(tooltip);
        
        // Set up the icon
        Icon toggleIcon = buffer.getIcon();
        toggle.setIcon(toggleIcon);
        
        // If this is the current buffer, make the button selected
        // and use a bold font
        if(view.getBuffer() == buffer) {
          toggle.setSelected(true);
          toggle.setFont(new Font(buttonFont.getName(), Font.BOLD, fontSize));
        }
        
        group.add(toggle);
        toggle.setActionCommand(buffer.getPath());
        toggle.addActionListener(actionHandler);
        
        this.add(toggle);
      }
  }
  
  
  public void bufferChanged() {
    this.bufferChanged(this.view.getBuffer());
  }
  
  
  public void bufferChanged(Buffer buffer) {
    Component[] component = this.getComponents();
    Font buttonFont = null;
    int fontSize = Integer.parseInt(
      jEdit.getProperty("buffer-selector.font-size")
      );
    
    for (int i = 0; i < component.length; i++) {
      if (component[i] instanceof AbstractButton) {
        AbstractButton toggle = ((AbstractButton)component[i]);
        if (buttonFont == null) {
          Font f = toggle.getFont();
          buttonFont = new Font(f.getName(), Font.PLAIN, fontSize);
        }
        
        if (toggle.getActionCommand().equals(buffer.getPath())) {
          toggle.setSelected(true);
          toggle.setFont(
            new Font(buttonFont.getName(), Font.BOLD, fontSize)
            );
        } else {
          toggle.setFont(buttonFont);
        }
      }
    }
  }
  
  
  /**
  * Remove all buttons from the <code>BufferSelector</code>, unregistering
  * them with the <code>ActionListener</code> first.
  */
  private void removeAllButtons() {
    // Unregister all buttons with the ActionListener
    Component[] component = this.getComponents();
    for (int i = 0; i < component.length; i++) {
      if (component[i] instanceof AbstractButton) {
        ((AbstractButton) component[i]).removeActionListener(
          this.actionHandler
          );
      }
    }
    // Remove all buttons from the BufferSelector
    this.removeAll();
  }
  
  
  /**
  * Adds event handlers associated with this Buffer Selector
  * <ul>
  * <li><em>View Resize Handler</em>: updates this when the View is resized
  * <li><em>Focus Handler</em>: added to all view edit panes, updates this
  * when an EditPane gains focus
  * </ul>
  */
  private void addHandlers() {
    this.view.addComponentListener(this.viewResizeHandler);
    
    EditPane[] editPanes = this.view.getEditPanes();
    for (int i = 0; i < editPanes.length; i++) {
      editPanes[i].getTextArea().addFocusListener(
        this.textAreaFocusHandler);
    }
  }
  
  
  /**
  * Removes event handlers associated with this Buffer Selector
  */
  private void removeHandlers() {
    this.view.removeComponentListener(this.viewResizeHandler);
    
    EditPane[] editPanes = this.view.getEditPanes();
    for (int i = 0; i < editPanes.length; i++) {
      editPanes[i].getTextArea().removeFocusListener(
        this.textAreaFocusHandler);
    }
  }
  
  
  /**
  * Tells if a <code>BufferSelector</code> is enabled for a specified
  * View
  */
  public static boolean isEnabledFor(View view) {
    return (bufferSelectors.get(view) != null);
  }
  
  
  /**
  * Adds a <code>BufferSelector</code> to a specified View
  */
  public static void addTo(View view) {
    BufferSelector bs = new BufferSelector(view);
    
    // Add View Resize Handlers and Text Area Handlers
    bs.removeHandlers(); // Necessary for EditPanes focus handlers
    bs.addHandlers();
    
    // Add the BufferSelector to the EditBus
    EditBus.addToBus(bs);
    
    // Adds a BufferSelector <-> View association
    bufferSelectors.put(view, bs);
    
    // Add the BufferSelector to the View
    if (BufferSelector.location == BufferSelector.TOP) {
      bs.topDocker.add();
    } else {
      bs.bottomDocker.add();
    }
    
    bs.update();
    view.invalidate();
    view.validate();
    
    //Funa edit
    view.repaint();
  }
  
  
  /**
  * Removes a <code>BufferSelector</code> from a specified View
  */
  public static void removeFrom(View view) {
    BufferSelector bs = (BufferSelector) bufferSelectors.get(view);
    if (bs != null) {
      Container cp = (Container) view.getContentPane();
      
      if (bs.bottomDocker.isVisible()) {
        bs.bottomDocker.remove();
      }
      
      if (bs.topDocker.isVisible()) {
        bs.topDocker.remove();
      }
      
      // Remove View Resize Handlers and Text Area Handlers
      bs.removeHandlers();
      // Remove from EditBus
      EditBus.removeFromBus(bs);
      // Removes the BufferSelector <-> View association
      bufferSelectors.remove(view);
      
      view.invalidate();
      view.validate();
      // Funa edit
      view.repaint();
      
    }
  }
  
  
  /**
  * Toggles a <code>BufferSelector</code> for a specified View
  */
  public static void toggleFor(View view) {
    if (BufferSelector.isEnabledFor(view)) {
      BufferSelector.removeFrom(view);
    } else {
      BufferSelector.addTo(view);
    }
  }
  
  
  public static void propertiesChanged() {
    int oldLocation = BufferSelector.location;
    BufferSelector.getProperties();
    View[] views = jEdit.getViews();
    
    for (int i = 0; i < views.length; i++) {
      if (BufferSelector.isEnabledFor(views[i])) {
        if (!defaultEnabled) {
          BufferSelector.removeFrom(views[i]);
        } else {
          if (oldLocation != BufferSelector.location) {
            BufferSelector.removeFrom(views[i]);
            BufferSelector.addTo(views[i]);
          }
        }
      } else {
        if (defaultEnabled) {
          BufferSelector.addTo(views[i]);
        }
      }
    }
    
    for (Enumeration e = bufferSelectors.elements(); e.hasMoreElements();) {
      BufferSelector bs = (BufferSelector) e.nextElement();
      bs.update();
    }
  }
  
  
  private static void getProperties() {
    defaultEnabled = jEdit.getBooleanProperty(
      "buffer-selector.default-enabled", false
      );
    location = jEdit.getIntegerProperty(
      "buffer-selector.location", BufferSelector.BOTTOM
      );
    
    minWidth = Integer.parseInt(jEdit.getProperty(
      "buffer-selector.min-width"
      ));
    maxWidth = Integer.parseInt(jEdit.getProperty(
      "buffer-selector.max-width"
      ));
    hgap = Integer.parseInt(jEdit.getProperty("button.hgap"));
    vgap = Integer.parseInt(jEdit.getProperty("button.vgap"));
  }
  
  
  private class ActionHandler implements ActionListener
  {
    public void actionPerformed(ActionEvent evt) {
      BufferSelector.this.view.setBuffer(
        jEdit.getBuffer(evt.getActionCommand())
        );
    }
  }
  
  
  // To keep track of the View being resized
  private class ViewResizeHandler extends ComponentAdapter
  {
    public void componentResized(ComponentEvent evt) {
      BufferSelector.this.update();
      BufferSelector.this.view.invalidate();
      BufferSelector.this.view.validate();
      
      // Funa edit
      BufferSelector.this.view.repaint();
    }
  }
  
  
  // BufferSelector acts on the last focused TextArea
  private class TextAreaFocusHandler extends FocusAdapter {
    public void focusGained(FocusEvent evt) {
      // walk up hierarchy, looking for an EditPane
      Component comp = (Component) evt.getSource();
      while (!(comp instanceof EditPane)) {
        if (comp == null) {
          return;
        }
        comp = comp.getParent();
      }
      // Funa edit
      BufferSelector.this.update();
      BufferSelector.this.bufferChanged(((EditPane) comp).getBuffer());
      BufferSelector.this.view.invalidate();
      BufferSelector.this.view.validate();
      // Funa edit
      BufferSelector.this.view.repaint();
    }
  }
  
  
  /** Responsible for docking BufferSelector at the right place */
  private interface Docker {
    boolean isVisible();
    
    void add();
    void remove();
  }
  
  
  private class ViewDocker implements Docker
  {
    private boolean visible = false;
    
    private int group = View.BOTTOM_GROUP;
    private int layer = View.BELOW_STATUS_BAR_LAYER;
    
    
    public ViewDocker(int group, int layer) {
      this.group = group;
      this.layer = layer;
    }
    
    
    public boolean isVisible() {
      return this.visible;
    }
    
    
    /**
    * Adds BufferSelector to the view
    */
    public void add() {
      this.visible = true;
      BufferSelector.this.view.addToolBar(
        this.group, this.layer, BufferSelector.this
        );
    }
    
    
    /**
    * Removes BufferSelector from the view
    */
    public void remove() {
      BufferSelector.this.view.removeToolBar(BufferSelector.this);
      this.visible = false;
    }
  }
}

